﻿using CryptoPro.Sharpei;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.IO;
using System.Numerics;
using System.Security;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;


namespace CryptoMAPI
{
    public class Lib
    {
        const string szOID_CP_GOST_R3410EL = "1.2.643.2.2.19";         //  Алгоритм ГОСТ Р 34.10 - 2001, используемый при экспорте / импорте ключей
        const string szOID_CP_GOST_R3410_12_256 = "1.2.643.7.1.1.1.1"; //  Алгоритм ГОСТ Р 34.10 - 2012 для ключей длины 256 бит, используемый при экспорте / импорте ключей
        const string szOID_CP_GOST_R3410_12_512 = "1.2.643.7.1.1.1.2"; //  Алгоритм ГОСТ Р 34.10 - 2012 для ключей длины 512 бит, используемый при экспорте / импорте ключей


        public static byte[] GetDigest3411(byte[] Data)
        {
            // Объект, реализующий алгритм хэширования.
            Gost3411CryptoServiceProvider GostHash = new Gost3411CryptoServiceProvider();
            return GostHash.ComputeHash(Data);
        }

        public static byte[] GetDigest2012_256(byte[] Data)
        {
            Gost3411_2012_256CryptoServiceProvider GostHash = new Gost3411_2012_256CryptoServiceProvider();
            return GostHash.ComputeHash(Data);
        } 

        public static byte[] GetDigest2012_512(byte[] Data)
        {  
            Gost3411_2012_512CryptoServiceProvider GostHash = new Gost3411_2012_512CryptoServiceProvider(); 
            return GostHash.ComputeHash(Data);
        }

        public static byte[] GetDigest(byte[] Data)
        { 
            return GetDigest3411(Data);
        }

        public static byte[] GetDigestByCertificate(byte[] Data, string FileCertificate)
        { 

            if (!File.Exists(FileCertificate))
            {
                throw new System.Exception("Не найден указанный файл с сертификатом: " + FileCertificate);
            }

            X509Certificate2 Certificate = new X509Certificate2();
            try
            {
                Certificate.Import(FileCertificate);
            }
            catch (System.Exception E)
            {
                throw new System.Exception("Ошибка при чтении сертификата: " + E.Message);
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410EL)
            {
                return GetDigest(Data );
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410_12_256)
            {
                return GetDigest2012_256(Data);
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410_12_512)
            {
                return GetDigest2012_512(Data);
            }
            throw new System.Exception("Не реализован алгоритм хеширования, открытый ключ: " + Certificate.PublicKey.Oid.Value.ToString());

        }

        public static X509Certificate2 GetX509Certificate(string SerialNumberHEX)
        {
            X509Certificate2 Certificate = null;
            X509Store store = new X509Store(StoreName.My, StoreLocation.CurrentUser);
            store.Open(OpenFlags.ReadOnly);
            X509Certificate2Collection coll = store.Certificates.Find(X509FindType.FindBySerialNumber, SerialNumberHEX, false);
            if (coll.Count == 1)
            {
                Certificate = coll[0];
            }
            store.Close();

            if (Certificate == null)
            {
                store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
                store.Open(OpenFlags.ReadOnly);
                coll = store.Certificates.Find(X509FindType.FindBySerialNumber, SerialNumberHEX, false);
                if (coll.Count == 1)
                {
                    Certificate = coll[0];
                }
                store.Close();
            }
            return Certificate;
        }

        public static string GetDictionaryValues(Dictionary<string, string> MAP)
        {
            string result = "";
            foreach (var pair in MAP.OrderBy(pair => pair.Key))
            {
                result = result + pair.Value;
            }
            return result;
        }
        
        private static byte[] GetSignature2001(byte[] Data, X509Certificate2 Certificate, string ContainerPassword)
        {
            if (Data.Length == 0)
            {
                throw new System.Exception("Нет данных для хеширования");
            }

            if (Certificate == null)
            {
                throw new System.Exception("Не найден сертификат");
            }

            byte[] baSign = null;

            CspKeyContainerInfo info = ((ICspAsymmetricAlgorithm)Certificate.PrivateKey).CspKeyContainerInfo;
            CspParameters cp = new CspParameters();

            Gost3410CryptoServiceProvider cert_key = Certificate.PrivateKey as Gost3410CryptoServiceProvider;

            cp.KeyContainerName = info.KeyContainerName;
            cp.ProviderType = info.ProviderType;
            cp.Flags = CspProviderFlags.UseExistingKey | CspProviderFlags.UseMachineKeyStore | CspProviderFlags.NoPrompt;

            if (null != cert_key)
            {
                cp.KeyContainerName = cert_key.CspKeyContainerInfo.KeyContainerName;
                cp.ProviderType = cert_key.CspKeyContainerInfo.ProviderType;
                cp.ProviderName = cert_key.CspKeyContainerInfo.ProviderName;
                cp.Flags = cert_key.CspKeyContainerInfo.MachineKeyStore
                                  ? (CspProviderFlags.UseExistingKey | CspProviderFlags.UseMachineKeyStore)
                                  : (CspProviderFlags.UseExistingKey);
            }
            if (ContainerPassword.Length > 0)
            {
                cp.KeyPassword = new SecureString();
                foreach (var c in ContainerPassword)
                {
                    cp.KeyPassword.AppendChar(c);
                }
            }

            Gost3410CryptoServiceProvider.UseMachineKeyStore = true;
            using (Gost3410CryptoServiceProvider csp = new Gost3410CryptoServiceProvider(cp))
            {
                Gost3411CryptoServiceProvider Hash = new Gost3411CryptoServiceProvider();
                baSign = csp.SignData(Data, Hash);

                // csp.VerifyData(Data, , baSign);
            }
            return baSign;
            /*
            if (Data.Length == 0)
            {
                throw new System.Exception("Нет данных для хеширования");
            } 

            if (Certificate == null)
            {
                throw new System.Exception("Не найден сертификат");
            } 

            byte[] HashValue = GetDigest3411(Data);

            // Объект, содержащий параметры криптопровайдера: 
            var cspParameters = new CspParameters();
            Gost3410CryptoServiceProvider cert_key = Certificate.PrivateKey as Gost3410CryptoServiceProvider;
            if (null != cert_key)
            {
                //копируем параметры csp из исходного контекста сертификата
                cspParameters.KeyContainerName = cert_key.CspKeyContainerInfo.KeyContainerName;
                cspParameters.ProviderType = cert_key.CspKeyContainerInfo.ProviderType;
                cspParameters.ProviderName = cert_key.CspKeyContainerInfo.ProviderName;
                cspParameters.Flags = cert_key.CspKeyContainerInfo.MachineKeyStore
                                  ? (CspProviderFlags.UseExistingKey | CspProviderFlags.UseMachineKeyStore)
                                  : (CspProviderFlags.UseExistingKey);

                if (ContainerPassword.Length > 0)
                {
                    cspParameters.KeyPassword = new SecureString();
                    foreach (var c in ContainerPassword)
                    {
                        cspParameters.KeyPassword.AppendChar(c);
                    }
                }
            }
            else
            {
                throw new System.Exception("Сертификат не содержит ссылку на закрытый ключ");
            }

            // Создаем криптопровайдер с заданными параметрами
            Gost3410CryptoServiceProvider gost = new Gost3410CryptoServiceProvider(cspParameters);
            // Создаем форматтер подписи для криптопровайдера
            GostSignatureFormatter GostFormatter = new GostSignatureFormatter(gost);
            // Создаем деформаттер подписи для криптопровайдера
            GostSignatureDeformatter GostDeformatter = new GostSignatureDeformatter(gost);
            // Подписываем хеш
            byte[] SignedHashValue = GostFormatter.CreateSignature(HashValue);
            bool ret = false;
            // Проверяем правильность подписи
            ret = GostDeformatter.VerifySignature(HashValue, SignedHashValue);
            var sg = SignedHashValue;//.Reverse().ToArray();  
            if (ret)
            {
                return sg;
            }
            return null;
            */
        }

        private static byte[] GetSignature2012_256(byte[] Data, X509Certificate2 Certificate, string ContainerPassword)
        {
            if (Data.Length == 0)
            {
                throw new System.Exception("Нет данных для хеширования");
            }

            if (Certificate == null)
            {
                throw new System.Exception("Не найден сертификат");
            }


            byte[] baSign = null;
            
                CspKeyContainerInfo info = ((ICspAsymmetricAlgorithm)Certificate.PrivateKey).CspKeyContainerInfo;
                CspParameters cp = new CspParameters();
             
               Gost3410_2012_256CryptoServiceProvider cert_key = Certificate.PrivateKey as Gost3410_2012_256CryptoServiceProvider;

                cp.KeyContainerName = info.KeyContainerName;
                cp.ProviderType = info.ProviderType;
                cp.Flags = CspProviderFlags.UseExistingKey  | CspProviderFlags.UseMachineKeyStore | CspProviderFlags.NoPrompt;

            if (null != cert_key)
            {
                cp.KeyContainerName = cert_key.CspKeyContainerInfo.KeyContainerName;
                cp.ProviderType = cert_key.CspKeyContainerInfo.ProviderType;
                cp.ProviderName = cert_key.CspKeyContainerInfo.ProviderName;
                cp.Flags = cert_key.CspKeyContainerInfo.MachineKeyStore
                                  ? (CspProviderFlags.UseExistingKey | CspProviderFlags.UseMachineKeyStore)
                                  : (CspProviderFlags.UseExistingKey);
            }
                if (ContainerPassword.Length > 0)
                {
                    cp.KeyPassword = new SecureString();
                    foreach (var c in ContainerPassword)
                    {
                        cp.KeyPassword.AppendChar(c);
                    }
                }

                Gost3410_2012_256CryptoServiceProvider.UseMachineKeyStore = true;
                using (Gost3410_2012_256CryptoServiceProvider csp = new Gost3410_2012_256CryptoServiceProvider(cp))
                {
                    Gost3411_2012_256CryptoServiceProvider Hash = new Gost3411_2012_256CryptoServiceProvider();
                    baSign = csp.SignData(Data, Hash);
                } 
            return baSign; 
        }

        private static byte[] GetSignature2012_512(byte[] Data, X509Certificate2 Certificate, string ContainerPassword)
        {
            if (Data.Length == 0)
            {
                throw new System.Exception("Нет данных для хеширования");
            }

            if (Certificate == null)
            {
                throw new System.Exception("Не найден сертификат");
            }

            byte[] baSign = null;

            CspKeyContainerInfo info = ((ICspAsymmetricAlgorithm)Certificate.PrivateKey).CspKeyContainerInfo;
            CspParameters cp = new CspParameters();

            Gost3410_2012_512CryptoServiceProvider cert_key = Certificate.PrivateKey as Gost3410_2012_512CryptoServiceProvider;

            cp.KeyContainerName = info.KeyContainerName;
            cp.ProviderType = info.ProviderType;
            cp.Flags = CspProviderFlags.UseExistingKey | CspProviderFlags.UseMachineKeyStore | CspProviderFlags.NoPrompt;

            if (null != cert_key)
            {
                cp.KeyContainerName = cert_key.CspKeyContainerInfo.KeyContainerName;
                cp.ProviderType = cert_key.CspKeyContainerInfo.ProviderType;
                cp.ProviderName = cert_key.CspKeyContainerInfo.ProviderName;
                cp.Flags = cert_key.CspKeyContainerInfo.MachineKeyStore
                                  ? (CspProviderFlags.UseExistingKey | CspProviderFlags.UseMachineKeyStore)
                                  : (CspProviderFlags.UseExistingKey);
            }
            if (ContainerPassword.Length > 0)
            {
                cp.KeyPassword = new SecureString();
                foreach (var c in ContainerPassword)
                {
                    cp.KeyPassword.AppendChar(c);
                }
            }

            Gost3410_2012_512CryptoServiceProvider.UseMachineKeyStore = true;
            using (Gost3410_2012_512CryptoServiceProvider csp = new Gost3410_2012_512CryptoServiceProvider(cp))
            {
                Gost3411_2012_512CryptoServiceProvider Hash = new Gost3411_2012_512CryptoServiceProvider();
                baSign = csp.SignData(Data, Hash);

               // csp.VerifyData(Data, , baSign);
            }
            return baSign;
        }

        public static byte[] GetSignature(byte[] Data, BigInteger SerialNumber, string ContainerPassword)
        {  
            if (Data.Length == 0)
            {
                throw new System.Exception("Нет данных для хеширования");
            }
            string FindSerial = SerialNumber.ToString("X");
            X509Certificate2 Certificate = GetX509Certificate(FindSerial);

            if (Certificate == null)
            {
                throw new System.Exception("Не найден указанный сертификат (серийный номер:" + SerialNumber.ToString() + ", в hex: " + FindSerial + ")");
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410EL)
            {
                return GetSignature2001(Data, Certificate, ContainerPassword);
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410_12_256)
            {
                return GetSignature2012_256(Data, Certificate, ContainerPassword);
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410_12_512)
            { 
                return GetSignature2012_512(Data, Certificate, ContainerPassword);
            }
            throw new System.Exception("Не реализован алгоритм подписи, открытый ключ: "+ Certificate.PublicKey.Oid.Value.ToString() + " (серийный номер:" + SerialNumber.ToString() + ", в hex: " + FindSerial + ")");
             
        }

        public static bool VerifySignature2001(byte[] Data, byte[] SignedValue, X509Certificate2 Certificate)
        { 
            // Объект, реализующий алгоритм ГОСТ 3410.
            Gost3410CryptoServiceProvider Gost = (Gost3410CryptoServiceProvider)Certificate.PublicKey.Key;
            // Объект, реализующий алгоритм хэширования ГОСТ 3411.
            Gost3411CryptoServiceProvider GostHash = new Gost3411CryptoServiceProvider();
            //  return Gost.VerifyData(Data, GostHash, SignedValue );
           // return Gost.VerifyData(Data, GostHash, SignedValue.Reverse().ToArray());
            return Gost.VerifyData(Data, GostHash, SignedValue.ToArray());
        }

        public static bool VerifySignature2012_256(byte[] Data, byte[] SignedValue, X509Certificate2 Certificate)
        { 
            Gost3410_2012_256CryptoServiceProvider Gost = (Gost3410_2012_256CryptoServiceProvider)Certificate.PublicKey.Key;
            Gost3411_2012_256CryptoServiceProvider GostHash = new Gost3411_2012_256CryptoServiceProvider(); 
            return Gost.VerifyData(Data, GostHash, SignedValue.ToArray());
        }

        public static bool VerifySignature2012_512(byte[] Data, byte[] SignedValue, X509Certificate2 Certificate)
        {
            Gost3410_2012_512CryptoServiceProvider Gost = (Gost3410_2012_512CryptoServiceProvider)Certificate.PublicKey.Key;
            Gost3411_2012_512CryptoServiceProvider GostHash = new Gost3411_2012_512CryptoServiceProvider();
            return Gost.VerifyData(Data, GostHash, SignedValue.ToArray());
        }

        public static bool VerifySignature(byte[] Data, byte[] SignedValue, string FileCertificate)
        {
            if (Data == null)
            {
                throw new System.Exception("Data is null");
            }
            if (SignedValue == null)
            {
                throw new System.Exception("SignedValue is null");
            }

            if (!File.Exists(FileCertificate))
            { 
                throw new System.Exception("Не найден указанный файл с сертификатом: "+ FileCertificate); 
            }

            X509Certificate2 Certificate = new X509Certificate2();
            try
            {
                Certificate.Import(FileCertificate);
            }
            catch (System.Exception E)
            {
                throw new System.Exception("Ошибка при чтении сертификата: " + E.Message);
            }


            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410EL)
            {
                return VerifySignature2001(Data, SignedValue, Certificate);
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410_12_256)
            {
                return VerifySignature2012_256(Data, SignedValue, Certificate);
            }

            if (Certificate.PublicKey.Oid.Value.ToString() == szOID_CP_GOST_R3410_12_512)
            {
                return VerifySignature2012_512(Data, SignedValue, Certificate);
            }
            throw new System.Exception("Не реализован алгоритм проверки для такого алгоритма: " + Certificate.PublicKey.Oid.Value.ToString() );
                        
        } 
    }
}
